#ifndef __OPENCL_VERSION__ //This should fail during an actual openCL compile, used only to trick Eclipse into syntax highlighting this file as "C" code.
#define __kernel
#define __global
#define kernel
#define global
#define constant
#define local
#define float2 float
#define int3 int
#define float3 float
#define float4 float
#define uchar4 char
#endif

/**
 * @return The 2D squared distance to a 2D rectangle at center (cx,cy) and lengths (Lx,Ly)?
 */
float distanceSquaredToRectangle2D(float x, float y, float cx, float cy, float Lx, float Ly)
{
	float dx_abs = fabs(x-cx);
	float dy_abs = fabs(y-cy);
	float halfLx = 0.5f*Lx;
	float halfLy = 0.5f*Ly;
	float distX = (dx_abs > halfLx) ? dx_abs - halfLx : 0.f;
	float distY = (dy_abs > halfLy) ? dy_abs - halfLy : 0.f;
	return distX*distX + distY*distY;
}

/** Determine if a point lies inside a 2D polygon.
 * From http://www.gamers.org/dEngine/rsc/usenet/comp.graphics.algorithms.faq
 * @param nVert Number of vertices in polygon.
 * @param poly A list of 2D points that describe the polygon vertices, up to length nPol.
 * @param point The point of interest.
 * @return True if the point is contained inside the polygon.
 * @author W. Randolph Franklin "Point Inclusion in Polygon Test".
 */
int containedInPoly2D(int nVert, float2 *poly, float2 point) {
	short i, j;
	bool c = false;
	for (i = 0, j = nVert-1; i < nVert; j = i++) {
		if ((((poly[i].y <= point.y) && (point.y < poly[j].y)) ||
			((poly[j].y <= point.y) && (point.y < poly[i].y))) &&
			(point.x < (poly[j].x - poly[i].x) * (point.y - poly[i].y) / (poly[j].y - poly[i].y) + poly[i].x))
				c = !c;
	}
	return c;
}

/**Compute the distance of a 3D point to a 3D circle oriented in its unit normal "n" direction going through
 * the circle's "center" point and given its "radius".
 * @param point The 3D point to test the distance to.
 * @param C Center of the circle.
 * @param n Normal unit vector of the circle's plane.
 * @param radius The circle's radius
 */
float distanceToCircle3D(const float3 point, const float3 C, const float3 n, const float radius, const bool bContainedOnly) {
	float3 CP = point - C; //Get the displacement vector between the point and the circle's center.
	float Pn = dot(n, CP);
	float Pr = length(CP - Pn*n);
	if (Pr <= radius) //Contained within the circle's radius, return just the perpendicular distance.
		return Pn;
	if (bContainedOnly) //Don't want distance, just whether it's contained or not
		return INFINITY;
	//Otherwise, calculate distance to the circle's edge
	Pr -= radius;
	return sqrt(Pr*Pr + Pn*Pn);
}

/*----------------------------------------------------------*//**
* Check for point containment (if direction==NULL) or line intersection (if direction!=NULL) with a list of volumes.
* Returns the distance to the ray intersection of the rear-face of the closest volume. at [0,180] degrees always positive.
* If the point lies within any of the volumes, then we must get the distance of the rear-face closest
* @param volumeParamBuffer A float stream containing floats/ints (all 4 byte aligned, very important!) information about all volumes serialized (concatenated) into a single "stream" buffer.
* @param cellAddressBuff An int buffer containing the cell's poly-group, points to the group address in the buffPolyGroups.
* @param buffPolyGroups An int buffer containing the indices of all the polygons [N, poly0, poly1, ..., polyN] in volumeParamBuffer.
* @param vParts Number of partitions in each dimension.
* @param vWidths Uniform width of a single partition in each dimension.
* @param globalCoords The 3D point to check if it resides inside the volumes list, or the eye of the ray tracing.
* 			Assumes that ALL meshes have been transformed into a local coordinate system, and the matrix must be provided to convert into that local system!
* @param srcMtxInv The 4x4 global->local transformation matrix for this single mesh volume.
* @param tolerance Maximum distance to consider a collision, usually used as an error tolerance.
* @param threshold Maximum distance for colormap.
* @return The distance away from the collision. 0 Means it was actually touching/contained with a polygon. Negative values indicate an error code (and should be a byte value) ranging from -128 to -1.
*//*-----------------------------------------------------------*/
float distanceToMesh(constant float *volumeParamBuffer, constant int *cellAddressBuff, constant int *buffPolyGroups,
		const int3 vParts, const float3 vWidths, const float3 globalCoords, constant float *srcMtxInv, const float fTolerance, const float fThreshold) {
	uint pPVB = 0;//The address
	float minDist = INFINITY;
	const int nVerticesShared = readInt(&pPVB, volumeParamBuffer);//If 0, then read each vertex count per face, otherwise assume this is the constant size for all
	const int nFaces = readInt(&pPVB, volumeParamBuffer);
	bool checkContained = true;
	const bool bContainedOnly = fThreshold <= 0.f; //Can we skip the distanceTo check?
	float3 localCoords = transformPoint_arrayVersion(srcMtxInv, globalCoords);//Convert global to local coordinates (in the meshes reference frame)
	float3 a,b,c;
	//Smart method, spatial indexing lets us check only a subset of polygons based on which cell the point lies in
	//Calculate which cell the point lies in. Must be converted into a 1D index
	int3 vInd;
	vInd.x = floor(localCoords.x / vWidths.x);
	vInd.y = floor(localCoords.y / vWidths.y);
	vInd.z = floor(localCoords.z / vWidths.z);

	if (vInd.x < 0 || vInd.y < 0 || vInd.z < 0 || vInd.x >= vParts.x || vInd.y >= vParts.y || vInd.z >= vParts.z)
		return INFINITY;//Early out, the point isn't inside the bounding box (boundary of the spatial index) which includes the threshold distance already so don't bother checking distance OR intersect!
	//Go based on sum of polygons through all 6 simple axis directions!

	const int baseIndID = vInd.x*vParts.z*vParts.y + vInd.y*vParts.z + vInd.z;//Convert 3D index into 1D
	const int baseGroupAddress = cellAddressBuff[baseIndID]; //Get the pointer to the poly-group
	//Check if it's contained first!
	int nVert = nVerticesShared;
	bool bOutside = true; //Does the point lie outside of the mesh (NOT contained).
	//If the point is inside an EMPTY partition, it could still be contained within the mesh volume so we MUST check the intersection by assuming a default direction
	int iEscapeDimDir = (baseGroupAddress == -1) ? 0 : buffPolyGroups[baseGroupAddress+1]; //Read EscapeDir (must skip polyCount).
	int3 indStep;// = (int3)(1,0,0); //For isContained, the unit direction vector to step when cycling through partitions for the ray to escape.
	dimDirToInt3(iEscapeDimDir, &indStep); //Computed in Java in a smart manner that follows the path of least resistance.
	const float3 vLine = convert_float3(indStep); //The float equivalent of indStep, used for ray-tracing/intersection methods to compute "isContained"Convert it into a float unit direction vector line
	int groupAddress = baseGroupAddress;//start form the base group, and step through all partitions until escaping the local domain.
	float fNormalDist = INFINITY;
	bool bIsOnNormalSide = false; //True if plane normal is facing toward the point of interest.
	while (checkContained) { //Do we also want to check if the points are contained?
		int nPolys = (groupAddress < 0) ? 0 : buffPolyGroups[groupAddress]; //Get the number of polygons in the cell group. If the address is -1, there are no polygons in current partition
		groupAddress++; //Skip over the polyCount and point to EscapeDir
		//Check intersections along which direction?
		for (int i = 0; i < nPolys; i++) { //Check only the faces in the cell group
			groupAddress++; //Go to next polygon (if first iteration, skips over escapeDir)
			uint p = pPVB + buffPolyGroups[groupAddress]; //Get pointer to the single polygon's address, after header
			//if (i*nVerticesShared*3 != p - pPVB)//simple unit test with just 1 cell
			if (nVerticesShared == 0) //For mixed polygon meshes
				nVert = readInt(&p, volumeParamBuffer);//Read the number of vertices for this polygon.
			if (p-pPVB > nFaces*nVerticesShared*3) //Something is wrong, read past the end of the polygon buffer
				return -2.f;
			bool bIntersects; //True if the half-ray intersects the next single triangle.
			float3 proj;
			switch (nVert) {
			case 3: //All triangles
				a = readFloat3(&p, volumeParamBuffer);
				b = readFloat3(&p, volumeParamBuffer);
				c = readFloat3(&p, volumeParamBuffer);
				bIntersects = intersectsTriangle3D(localCoords, vLine, a, b, c, &proj, &fNormalDist, &bIsOnNormalSide); //Intersects this triangle?
			break;
			default:
				bIntersects = false; //Ignore this unknown volume
			break;
			}
			if (bIntersects) { //The point intersected the polygon
				if (fabs(fNormalDist) < fabs(minDist)) { //Are we the closest polygon?
					minDist = fNormalDist; //Save the bew closest distance
					//NEW: If the point lies on the opposite side of the polygon's normal direction, assume we're INSIDE the mesh
					bOutside = bIsOnNormalSide;
				}
				checkContained = false; //Don't check any more partitions
				//no break; We still need to check the rest of the polygons, but JUST in this cell
			}
		}
		//Should we continue to the next iteration?
		while (true) { //Step until we find a populated partition:
			vInd += indStep; //Next vInd as we traverse across the half domain along the path of least resistance (+= fails, wtf??)
			if (vInd.x < 0 || vInd.y < 0 || vInd.z < 0 || vInd.x >= vParts.x || vInd.y >= vParts.y || vInd.z >= vParts.z) {
				checkContained = false; //exit the outer "contained" loop
				break; //Escaped the domain
			}
			int indID = vInd.x*vParts.z*vParts.y + vInd.y*vParts.z + vInd.z;//Convert 3D index into 1D
			groupAddress = cellAddressBuff[indID];//Get the pointer to the length of the cell group
			if (groupAddress != -1)
				break; //Found a cell that has polygons in it, allow it to be computed
		}
	} //End partition loop while searching for intersections
	if (!bOutside) //The closest polygon intersected, was from the inside of the mesh
		return 0.f;
	else if (bContainedOnly)
		return INFINITY; //NOT contained, and we don't care about the distance
	//--------------------------
	//Was NOT contained, so check how far away it is:
	if (baseGroupAddress == -1)
		return INFINITY; //Early out, the point lies in a cell containing NO polygons, any neighboring poly's reside greater than threshold distance away.
	groupAddress = baseGroupAddress; //Restart computation from the baseGroupAddress
	int nPolys = buffPolyGroups[groupAddress++];//Get the number of polygons in the cell group, then jump NEXT to escapeDir
	for (int i = 0; i < nPolys; i++) { //Check only the faces in the cell group
		groupAddress++; //Go to NEXT polygon. This is the index in the group list, pointing to a list of polygon indices. (skips the length, and escapeDimDir)
		uint p = buffPolyGroups[groupAddress] + pPVB; //Get pointer to the single polygon's address, after header
		//if (i*nVerticesShared*3 != p - pPVB)//simple unit test with just 1 cell
		if (nVerticesShared == 0) //For mixed polygon meshes
			nVert = readInt(&p, volumeParamBuffer);//Read the number of vertices for this polygon.
		if (p-pPVB > nFaces*nVerticesShared*3) //Something is wrong, read past the end of the polygon buffer
			return -2.f;
		switch (nVert) {
		case 1: //All points (point-point collisions)
			a = readFloat3(&p, volumeParamBuffer); //Read a single point
			minDist = min(length(localCoords - a), minDist); //Distance to closest collision among ALL points!
		break;
		case 2: //All lines (point-line collisions)
			a = readFloat3(&p, volumeParamBuffer); //Read start point
			b = readFloat3(&p, volumeParamBuffer); //Read end point
			minDist = min(distanceToSegment3D(localCoords, a, b), minDist); //Distance to closest collision among ALL lines!
		break;
		case 3: //All triangles
			a = readFloat3(&p, volumeParamBuffer);
			b = readFloat3(&p, volumeParamBuffer);
			c = readFloat3(&p, volumeParamBuffer);
			minDist = min(distanceToTriangle3D(localCoords, a, b, c, fThreshold), minDist); //Distance to closest collision among ALL polygons!
			//minDist = distanceToTriangle3D(localCoords, a, b, c, fThreshold); //Distance to current polygon
		break;
		default:
			//Ignore this volume
		break;
		}
		if (minDist <= fTolerance)
			break; //Within error tolerance, considered colliding, so skip the remaining polygons
		/*if (minDist < fThreshold)//NOTE this early out is faster, but it might not find the CLOSEST collision distance
			return minDist;//early-out, already "touches" the mesh, otherwise continue cycling through polygons*/
	}
	return minDist * srcMtxInv[32]; //Multiply by the scale factor;//No collision detected closer than threshold*/
}

/**This version complies with Java3D's geometry which requires the cylinder's axis along the y direction (NOT z) with the origin CENTERED on that axis (NOT at the base).
 * @param pParams is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data, providing half height and radius.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [inverse, forward, scalar]
 * @return distance (in meters) to a regular cylinder.
 */
float distanceToCylinder(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;//Pointer for volumeParamBuffer[p]
	float3 point = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)

	float fPRad = sqrt(point.x*point.x+point.z*point.z);//The projected radial distance on the cylinder's XZ plane
	float fDist; //The final return value will be stored here
	float fHalfHeight	= readFloat(&p, volumeParamBuffer); //Average of inner/outer radius
	float fRadius		= readFloat(&p, volumeParamBuffer); //Half-width of rectangle cross-section along the Y-axis
	if (fPRad <= fRadius) { //Contained within the radius
		if (point.y < -fHalfHeight) //Lies below the cylinder
			fDist = -fHalfHeight - point.y; //Return the normal distance to the circle
		else if (point.y > fHalfHeight) //Lies above the cylinder
			fDist = point.y - fHalfHeight; //Return the normal distance to the circle
		else { //Contained
			return 0.f;
		}
		if (bContainedOnly)
			return INFINITY; //Was NOT Contained and we don't need the distance
	}
	else { //Outside of the radius
		if (bContainedOnly)
			return INFINITY; //NOT contained, and only asked for containment
		fPRad -= fRadius; //How far from the surface in radial direction?
		if (point.y < -fHalfHeight) { //Lies below the cylinder
			float dy = -fHalfHeight - point.y;
			fDist = sqrt(dy*dy + fPRad*fPRad); //Return the normal&radius distance to the circle's edge
		}
		else if (point.y > fHalfHeight) { //Lies above the cylinder
			float dy = point.y - fHalfHeight;
			fDist = sqrt(dy*dy + fPRad*fPRad);  //Return the normal&radius distance to the circle's edge
		}
		else //To the side of the cylinder
			fDist = fPRad;
	}
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}

/**
 * @param volumeParamBuffer is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data, providing half height and radius.
 * @param globalCoords Input coordinates are global.
 * @param srcMtxInv The matrix package: [inv(gl) mtx, fwd(lg) mtx, scalar]
 * @param bContainedOnly If true, performs an early out that won't calculate the distance, simply returns 0 or infinity in that case.
 * @return distance (in meters) to a rectangular box.
 */
float distanceToBox(constant float *volumeParamBuffer, const float3 globalCoords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;//Pointer for volumeParamBuffer[p]
	float3 point = transformPoint_arrayVersion(srcMtxInv, globalCoords); //Convert global to local coordinates (in the volume's reference frame)
	float fHalfX	= readFloat(&p, volumeParamBuffer); //The half length in x-direction
	float fHalfZ	= readFloat(&p, volumeParamBuffer); //The half length in z-direction
	float fHalfH	= readFloat(&p, volumeParamBuffer); //The half height (y-direction)
	float3 low  = (float3)(-fHalfX, -fHalfH, -fHalfZ);
	float3 high = (float3)( fHalfX,  fHalfH,  fHalfZ);
	float du = ((point.x>high.x)||(point.x<low.x)) ? min(fabs(point.x-high.x), fabs(point.x-low.x)) : 0.f;
	float dv = ((point.z>high.z)||(point.z<low.z)) ? min(fabs(point.z-high.z), fabs(point.z-low.z)) : 0.f;
	float dn = ((point.y>high.y)||(point.y<low.y)) ? min(fabs(point.y-high.y), fabs(point.y-low.y)) : 0.f;
	if (du == 0.f && dv == 0.f && dn == 0.f) //Contained!
		return 0.f;
	if (bContainedOnly)
		return INFINITY; //Was NOT Contained and we don't need the distance
	float fDist = sqrt(du*du + dv*dv + dn*dn); //The final return value will be stored here
	return fDist;// * srcMtxInv[32]; //Multiply by the scale factor
}

/**This version requires the cylinder's axis along the z direction with the origin centered on the bottom face (z=0)
 * @param pParams is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [inverse, forward, scalar]
 * @return distance (in meters) to a regular cylinder.
 */
float distanceToCylinderZ(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;//Pointer for volumeParamBuffer[p]
	float3 point = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)

	float fPRad = sqrt(point.x*point.x+point.y*point.y);//The projected radial distance on the cylinder's plane
	float fDist; //The final return value will be stored here
	float fHeight	= readFloat(&p, volumeParamBuffer); //Average of inner/outer radius
	float fRadius	= readFloat(&p, volumeParamBuffer); //Half-width of rectangle cross-section
	if (fPRad <= fRadius) { //Contained within the radius
		if (point.z < 0) //Lies below the cylinder
			fDist = -point.z; //Return the normal distance to the circle
		else if (point.z > fHeight) //Lies above the cylinder
			fDist = point.z - fHeight; //Return the normal distance to the circle
		else { //Contained
			return 0.f;
		}
		if (bContainedOnly)
			return INFINITY; //Was NOT Contained and we don't need the distance
	}
	else { //Outside of the radius
		if (bContainedOnly)
			return INFINITY; //NOT contained, and only asked for containment
		fPRad -= fRadius; //How far from the surface in radial direction?
		if (point.z < 0) //Lies below the cylinder
			fDist = sqrt(point.z*point.z + fPRad*fPRad); //Return the normal&radius distance to the circle's edge
		else if (point.z > fHeight) { //Lies above the cylinder
			float dz = point.z - fHeight;
			fDist = sqrt(dz*dz + fPRad*fPRad);  //Return the normal&radius distance to the circle's edge
		}
		else //To the side of the cylinder
			fDist = fPRad;
	}
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}

/**
 * Check if a point collides with a point cloud.
 * @param volumeParamBuffer The start of this cloud volume's param buffer.
 * @param coords The 3D point (should be global) to check collision distance with a list of cloud points.
 */
float distanceToCloud(constant float *volumeParamBuffer, const float3 coords) {
	uint p = 0;
	float fDist = INFINITY;
	int numPoints = readInt(&p, volumeParamBuffer); //read 32-bit integer, containing type (class_id) of current volume
	for (int i = 0; i < numPoints; i++) {
		float3 point = readFloat3(&p, volumeParamBuffer);
		fDist = min(length(point - coords), fDist);
	}
	return fDist;
}

/**
 * @param pParams is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [inverse, forward, scalar]
 * @return distance (in meters) to a sphere portion (a sphere sliced off by a single plane).
 */
float distanceToPortionSphere(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;
	float3 point = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)
	float fRadiusBaseLocal = readFloat(&p, volumeParamBuffer);

	//Calculation of the scalar product of the OP vector and the dish axis
	if (point.z < 0.f) { //The point is below the sphere's BB, NOT contained, calculate distance away form plane
		if (bContainedOnly)
			return INFINITY; //NOT contained, and only asked for containment
		float d2_xy = point.x*point.x+point.y*point.y;
		float d_xy = max(sqrt(d2_xy) - fRadiusBaseLocal, 0.f);
		return sqrt(point.z*point.z+d_xy*d_xy);
	}
	float radiusSphere = readFloat(&p, volumeParamBuffer); //m_fRadiusSphereLocal;
	float pCenterCap = readFloat(&p, volumeParamBuffer); //The distance from cap's center to sphere's center (along local z-axis)
//	float3 pCenterSphereLocal = readFloat3(&p, volumeParamBuffer); //m_fCenterSphereLocal

	float fDist = length(point - (float3)(0.f, 0.f, pCenterCap));
//	float fDist = length(point - pCenterSphereLocal);

	if (fDist < radiusSphere) //Fully contained
		return bContainedOnly ? 0.f : INFINITY; //Contained or NOT.
	else
		fDist -= radiusSphere;
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}

/**
 * @param pParams is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [inverse, forward, scalar]
 * @return Distance (in meters) to a dish (half 3D ellipsoid).
 */
float distanceToDish(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;
	float3 point = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)
	float fRadius = readFloat(&p, volumeParamBuffer);
	float d2_xy = point.x*point.x+point.y*point.y;
	//Calculation of the scalar product of the OP vector and the dish axis
	if (point.z < 0.f) { //The point is below the sphere's BB
		float d_xy = max(sqrt(d2_xy) - fRadius, 0.f);
		return sqrt(point.z*point.z+d_xy*d_xy);
	}
	float R2 = fRadius*fRadius;
	float fHeight = readFloat(&p, volumeParamBuffer);

	if (d2_xy/R2 + (point.z*point.z)/(fHeight*fHeight) <= 1.f) {
		return 0.f; //The point is contained inside the dish
	}
	else if (bContainedOnly)
		return INFINITY; //NOT contained, and only asked for containment
	float d = sqrt(d2_xy + point.z*point.z);
	float zd2 = pow(point.z/d, 2);
	float fDist = (1-zd2)*pow(d-fRadius, 2) + zd2*pow(d-fHeight, 2);
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}

/**@param volumeParamBuffer is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [4x4inverse, 4x4forward, scalar]
 * @return Distance (in meters) to a pyramid.
 */
float distanceToPyramid(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;
	float3 PLocal	= transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)
	float fHalfX	= readFloat(&p, volumeParamBuffer)/2.f;
	float fHalfY	= readFloat(&p, volumeParamBuffer)/2.f;
	float fHeight	= readFloat(&p, volumeParamBuffer);
	float fHalfZ	= fHeight/2.f;

	float fBottomX	= readFloat(&p, volumeParamBuffer);
	float fBottomY	= readFloat(&p, volumeParamBuffer);
	float fTopX		= readFloat(&p, volumeParamBuffer);
	float fTopY		= readFloat(&p, volumeParamBuffer);

	float scalU = PLocal.z + fHalfZ;
	float fDist;
	if ((scalU < 0) || (scalU > fHeight)) {
		if (bContainedOnly)
			return INFINITY; //NOT contained, and only asked for containment
		//The point is out of bounds
		//The distance to the object is the distance to the edge of the object on the affected end
		if (scalU < 0) { //Below
			float dist2_xy = distanceSquaredToRectangle2D(PLocal.x, PLocal.y, -fHalfX, -fHalfY, fBottomX, fBottomY);
			float dist2_z = (PLocal.z + fHalfZ)*(PLocal.z + fHalfZ);
			fDist = sqrt(dist2_xy+dist2_z);
		}
		else { //Above
			float dist2_xy = distanceSquaredToRectangle2D(PLocal.x, PLocal.y, fHalfX, fHalfY, fTopX, fTopY);
			float dist2_z = (PLocal.z - fHalfZ)*(PLocal.z - fHalfZ);
			fDist = sqrt(dist2_xy+dist2_z);
		}
	}
	else {
		//TODO: APPROXIMATE VALUE! Apply THALES method:
		float ratio = scalU/fHeight;
		float x = fBottomX + ratio*(fTopX-fBottomX);
		float y = fBottomY + ratio*(fTopY-fBottomY);
		float cx = -fHalfX + ratio*(2*fHalfX);
		float cy = -fHalfY + ratio*(2*fHalfY);
		float dist2_xy = distanceSquaredToRectangle2D(PLocal.x, PLocal.y, cx, cy, x, y);
		fDist = sqrt(dist2_xy);
	}
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}

/**@param volumeParamBuffer is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv TODO: Need to update this extruded polygon to use a matrix package: [4x4inverse, 4x4forward, scalar]
 * @return Distance (in meters) to an extruded polygon.
 */
float distanceToExtPoly(constant float *volumeParamBuffer, const float3 globalCoords, constant float* srcMtxInv, const bool bContainedOnly) {
	const float3 PLocal	= transformPoint_arrayVersion(srcMtxInv, globalCoords); //Convert global to local coordinates (in the volume's reference frame)
	uint p = 0;//Assume incoming pointer is already at the head of the volume stream.
	float height = readFloat(&p, volumeParamBuffer);//Read the overall polygon height before any sub-polygon headers
	const struct CPolyHeader polyHeader = readPolyHeader(volumeParamBuffer+p);//Don't forget the memory offset!
	float3 OP = PLocal - polyHeader.O; //Convert global to local origin offset
	//Rotate global to local coordinate system:
	float scalU = dot(OP, polyHeader.U);
	float scalV = dot(OP, polyHeader.V);
	float scalN = dot(OP, polyHeader.N);

	float3 pointProj = polyHeader.O + scalU*polyHeader.U + scalV*polyHeader.V; //No N component here! We're working with the bottom face that intersects the origin
	//Check if the point lies along the bottom face in a 2D manner:
	bool bContainedInPoly = belongsToPolygon(pointProj, polyHeader, true); //Can't use "globalCoords" here, must be pointProj!
	float dn; //The normal (z) component distance from polygon
	if (scalN < 0.f)
		dn = -scalN;
	else if (scalN > height)
		dn = scalN - height;
	else { //Within the z-axis
		dn = 0.f;
		if (bContainedOnly) //We don't care about the distanceTo?
			return bContainedInPoly ? 0.f : INFINITY; //Contained or NOT.
	}
	float fDist = INFINITY;
	//The point may lie outside the polygon, so we must calculate the distance
	float2 point = (float2)(scalU, scalV); //Get the 2D projection
	//The distance from the point to the polygon is therefore the minimum distance to the edges of the polygon
	float3 P0, P1, P2;
	p = 0;//reset the stream pointer
	P1 = readFloat3(&p, polyHeader.afterHeaderBuffer) - polyHeader.O; //The very first vertex location
	P1 = (float3)(dot(P1, polyHeader.U), dot(P1, polyHeader.V), 0.f);
	P0 = P1; //Setup for first iteration of loop
	for (int i = 0; i < polyHeader.nVertices; i++) {
		float3 edgeUnitVect	= readFloat3(&p, polyHeader.afterHeaderBuffer); //Edge unit direction vector
		float edgeLen		= readFloat (&p, polyHeader.afterHeaderBuffer); //Edge length
		if (i < polyHeader.nVertices-1) {
			P2				= readFloat3(&p, polyHeader.afterHeaderBuffer) - polyHeader.O; //The NEXT Vertex location
			P2 = (float3)(dot(P2, polyHeader.U), dot(P2, polyHeader.V), 0.f);
		}
		else
			P2 = P0; //Loop back to the FIRST vertex again
		//Keep the smallest distance to each polygon line
		fDist = min(fDist, distanceToSegment2D(point, (float2)(P1.x, P1.y), (float2)(P2.x, P2.y)));
		P1 = P2; //Prepare for the next iteration
	}
	if (bContainedInPoly) //Was it contained inside the 2D base polygon?
		fDist = dn; //Return the height z-component only
	else { //Return the z-component combined with the x-y component
		if (dn != 0.f) //Only need to compute if non-zero z-distance, otherwise fDist is already the xy-distance.
			fDist = sqrt(fDist*fDist + dn*dn);
	}
	return fDist;
}

/**@param volumeParamBuffer is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [4x4inverse, 4x4forward, scalar]
 * @return Distance (in meters) to a Circular Torus.
 */
float distanceToCircularTorus(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;//Pointer for volumeParamBuffer[p]
	float3 pProj = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)

	float dXY = sqrt(pProj.x*pProj.x+pProj.y*pProj.y);
	float xProjXYUnit = pProj.x/dXY;
	float yProjXYUnit = pProj.y/dXY;

	float anglePproj = acos(xProjXYUnit);	//Angle between 0 and PI
	if (yProjXYUnit < 0) anglePproj = -anglePproj; //Negate angle if y<0

	//Periodic in angle keeps it between 0-2Pi
	if (anglePproj < 0) 			anglePproj += 2*M_PI;
	else if (anglePproj > 2*M_PI) 	anglePproj -= 2*M_PI;

	float fDist; //The final return value will be stored here
	float m_angle				= readFloat(&p, volumeParamBuffer);
	float m_aveR				= readFloat(&p, volumeParamBuffer);
	float m_calculatedRadius	= readFloat(&p, volumeParamBuffer);
	//m_angle = fmod(m_angle, 2.f*(float)M_PI); //clamp to angle between 0-2*Pi
	if (m_angle < 0)			m_angle += 2*M_PI;
	else if (m_angle > 2*M_PI)	m_angle -= 2*M_PI;

	if ((anglePproj >= 0) && (anglePproj <= m_angle)) { //The point lies within the desired sector (0-m_angle)
		float c = xProjXYUnit; //cos
		float s = yProjXYUnit; //sin
		float3 C = (float3)(m_aveR*c, m_aveR*s, 0.f);//Ave radius (average of outer and inner radius)
		//float distToC = sqrt((pProj.x-C.x)*(pProj.x-C.x)+(pProj.y-C.y)*(pProj.y-C.y)+(pProj.z-C.z)*(pProj.z-C.z));
		float distToC = length(pProj - C);
		if (distToC <= m_calculatedRadius) //Contained?
			return 0.f; //Fully contained
		else if (bContainedOnly) //Not contained, and don't care about the distance
			return INFINITY; //Only asked for containment
		fDist = distToC - m_calculatedRadius;
	}
	else { //Point lies outside the m_angle sector.
		if (bContainedOnly)
			return INFINITY; //NOT contained, and only asked for containment
		float3 n; //Normal vector of circular cross-section face
		float3 C; //Center point of circular cross-section face
		if (anglePproj >= m_angle/2.f + M_PI) { //The point lies closer to the first circular face
			n = readFloat3(&p, volumeParamBuffer);//m_calculatedN1
			C = readFloat3(&p, volumeParamBuffer);//m_calculatedC1
		}
		else {
			p += 6; //skips reading N1,C1
			n = readFloat3(&p, volumeParamBuffer);//m_calculatedN2
			C = readFloat3(&p, volumeParamBuffer);//m_calculatedC2
		}
		float3 PprojSurTranche = projectionSurPlan(pProj, C, n/*, false*/);
		float distToC = length(PprojSurTranche - C);

		if (distToC <= m_calculatedRadius) { //Use the perpendicular distance
			fDist = length(PprojSurTranche - pProj);
		}
		else { //Need the distance to the face's circular edge
			float3 V = PprojSurTranche - C;//unit vector from C in the direction of P on the plane surface
			V = normalize(V);
			//float3 pointSurCercle = (float3)(C.x + m_calculatedRadius*V.x, C.y + m_calculatedRadius*V.y, C.z + m_calculatedRadius*V.z);
			float3 pointSurCercle = C + m_calculatedRadius*V;
			fDist = length(pointSurCercle - pProj);
		}
	}
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}
/**@param volumeParamBuffer is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [4x4inverse, 4x4forward, scalar]
 * @return Distance (in meters) to a Rectangular Torus.
 */
float distanceToRectangularTorus(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;//Pointer for volumeParamBuffer[p]
	float3 point = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame)

	float dXY = sqrt(point.x*point.x+point.y*point.y);
	float xProjXYUnit = point.x/dXY;
	float yProjXYUnit = point.y/dXY;

	float anglePproj = acos(xProjXYUnit);	//Angle between 0 and PI
	if (yProjXYUnit < 0)	anglePproj = -anglePproj; //Negate angle if y<0

	//Periodic angle between 0-2Pi:
	if (anglePproj < 0.f)			anglePproj += 2*M_PI;
	else if (anglePproj > 2*M_PI)	anglePproj -= 2*M_PI;

	float fDist; //The final return value will be stored here
	float m_angle				= readFloat(&p, volumeParamBuffer); //Volume's sweep angle
	float m_aveR				= readFloat(&p, volumeParamBuffer); //Average of inner/outer radius
	float m_calculatedHalfWidth	= readFloat(&p, volumeParamBuffer); //Half-width of rectangle cross-section
	float m_calculatedHalfH		= readFloat(&p, volumeParamBuffer); //Half-height of rectangle cross-section
	//Periodic angle between 0-2*Pi
	if (m_angle < 0)			m_angle += 2*M_PI;
	else if (m_angle > 2*M_PI)	m_angle -= 2*M_PI;

	if ((anglePproj >= 0.f) && (anglePproj <= m_angle)) { //The point lies within the desired sector (0-m_angle)
		float c = xProjXYUnit; //cos
		float s = yProjXYUnit; //sin
		float3 C = (float3)(m_aveR*c, m_aveR*s, 0.f);//Ave radius (average of outer and inner radius) at the half-height
		float3 CP = point - C;
		float dz = fabs(CP.z) - m_calculatedHalfH;
		float dr = sqrt(CP.x*CP.x + CP.y*CP.y) - m_calculatedHalfWidth;
		if (dr <= 0.f) { //Contained?
			if (dz <= 0.f) //Lies above/below, just need the z-distance component
				return 0.f; //Fully contained
			else if (bContainedOnly)
				return INFINITY; //NOT contained, and only asked for containment
			fDist = dz;
		}
		else {//Point lies at a larger/smaller radius than the torus, but within the m_angle sector
			if (bContainedOnly)
				return INFINITY; //NOT contained, and only asked for containment
			else if (dz > 0.f) //Lies above/below, just need the z-distance component
				fDist = sqrt(dz*dz + dr*dr);
			else
				fDist = dr;
		}
	}
	else { //Point lies outside the m_angle sector.
		if (bContainedOnly)
			return INFINITY; //NOT contained, and only asked for containment
		float3 n; //normal to the rectangular cross-section face.
		float3 C; //center to the rectangular cross-section face.
		if (anglePproj >= m_angle/2.f + M_PI) { //The point lies closer to the first rectangular face
			n = readFloat3(&p, volumeParamBuffer); //m_calculatedN1
			C = readFloat3(&p, volumeParamBuffer); //m_calculatedC1
		}
		else {
			p += 6; //skips reading N1,C1
			n = readFloat3(&p, volumeParamBuffer); //m_calculatedN2
			C = readFloat3(&p, volumeParamBuffer); //m_calculatedC2
		}
		float3 CP = point - C; //Displacement between rectangle's center and point of interest.
		//Now lets decompose CP into its 3 components in a cylindrical coordinate system centered on the winning rectangular cross section:
		//float CP.z; //Component of CP in the z-direction
		float dn = dot(CP,n); //Component of CP in the n direction (angular-direction)
		float3 CPr = CP - (float3)(0.f, 0.f, CP.z) - n*dn; //The component of CP in the radial-direction
		float dr = length(CPr) - m_calculatedHalfWidth;
		float dz = fabs(CP.z) - m_calculatedHalfH;
		if (dr <= 0.f) { //contained within radial bound
			if (dz <= 0.f) //contained within the z-bound
				fDist = fabs(dn); //just need plane-perpendicular distance
			else
				fDist = sqrt(dz*dz + dn*dn); //just need edge distance
		}
		else { //Lies outside of radial bound
			if (dz <= 0.f) //contained within the z-bound
				fDist = sqrt(dr*dr + dn*dn);
			else //Lies on a corner (outside of the radial and z bounds)
				fDist = sqrt(dr*dr + dz*dz + dn*dn);
		}
	}
	return fDist * srcMtxInv[32]; //Multiply by the scale factor
}

/**WARNING: For sheared circle faces, the distance calculation isn't working perfectly for near the circular faces. Contains works perfectly in all cases.
 * @param volumeParamBuffer is a pointer to the volumeParamBuffer that should be pointing to the relevant stream data.
 * @param coords Input coordinates are global.
 * @param srcMtxInv The matrix package: [4x4inverse, 4x4forward, scalar]
 * @param bCheckContain If true performs an early-out that only checks for contained (returns INFINITY if NOT contained).
 * 			When false, returns the distance to the cone.
 * @return Distance (in meters) to a generic cone (snout, sloped cylinder, etc.).
 * Always returns 0.f when contained, but if bCheckContain==true then skips the detailed distance computation and returns INFINITY.
 */
float distanceToCone(constant float *volumeParamBuffer, const float3 coords, constant float* srcMtxInv, const bool bContainedOnly) {
	uint p = 0;//Pointer for volumeParamBuffer[p]
	float3 point = transformPoint_arrayVersion(srcMtxInv, coords); //Convert global to local coordinates (in the volume's reference frame).
	//The top circle's center (CTop) defines the displacement "axis" between the 2 centers.
	float3 CTop	= readFloat3(&p, volumeParamBuffer); //(m_xOffset, m_yOffset, m_height)
	point += CTop/2.f; //Convert our coordinate system relative to the two-circle-axis center: the origin of the matrix transform is at the axis center, not the center of the bottom circle.
	//NOTE: the the bottom circle's center is the NOW the origin in local 3D XYZ (and 2D UV) system.
	float3 v = CTop; //The axis between the center's of both circles.
	float lenV	= length(v);
	if (lenV == 0.f)  //Axis has no length, assume it's a point
		return length(coords) * srcMtxInv[32]; //convert to absolute distance
	v /= lenV; //Normalize it to a unit vector
	if (length(point) == 0.f)
		return 0.f; //Point lies right on bottom circle's center, early out contained.
	float3 w = cross(v, point); //Compute the w-axis (everything will projected onto the UV plane, so all w components will be zero by 2D construction)
	float lenW = length(w);
	if (lenW != 0.f)
		w /= lenW; //TODO: in the special case that lenW=0, the point lies along the v axis, thus can skip straight to the top/bottom circle distance calculation if the contains fails
	float3 u = cross(v, w);
	float lenU = length(u);
	//if (lenU == 0.f) Impossible unless lenW=0
	u /= lenU;//Normalize the u-axis (points along the bottom radius!)
	//Project the point onto the 2D UV plane
	float2 CPuv = (float2)(dot(point, u), dot(point, v));
	float2 Cuv  = (float2)(0.f, lenV); //Top circle's center in UV coordinates

	float  rad[2];//scalar radius for top and bottom faces (this is the radius for this plane ONLY, when shear exists the top/bottom face is NOT a circle anymore).
	float3 R[2]; //Will store the Radius generated from the circle normal that lies in the r-z plane
	float3 n[2]; //The 3D circle normals
	float2 Ruv[2]; //The 2D UV projection of the 3D radius (R).
	for (int i = 0; i < 2; i++) {//index: 0=bottom, 1=top circle
		n[i]	= readFloat3(&p, volumeParamBuffer); //the i'th circles normal vector
		rad[i]	= readFloat (&p, volumeParamBuffer); //The scalar radius of the i'th circle
		R[i] = u * rad[i]; //"Normalize" to a unit vector, then scale that to the circle's radius in 3D
		intersectPlaneLine((float3)(0.f, 0.f, 0.f), n[i], R[i], v, false, R + i, NULL, NULL); //Project the radius point onto the shear n-plane, along direction v-axis.
		//Project the radius vector into the 2D UV plane:
		Ruv[i] = (float2)(dot(R[i],u), dot(R[i],v)); //Here the w-component is zero by construction (or at least it should be)
		Ruv[i] /= sign(Ruv[i].x); //Flip it so that the x component is always positive (and alters y accordingly).
	}
	#define NUMVERT 4
	float2 polyUV[NUMVERT]; 	//Create the ABCD points of the 2D quad in UV coordinates. They will ALWAYS be oriented the same way in 2D, by construction!
	polyUV[0] = -Ruv[0];		//a (bottom-left corner)
	polyUV[1] = Cuv - Ruv[1];	//b (top-left corner)
	//By construction, the point will always be on the A-B side of the v-axis, not on the C-D side, But we still need all 4 points for a "contains" operation!
	polyUV[2] = Cuv + Ruv[1];	//c (top-right corner)
	polyUV[3] = Ruv[0];			//d (bottom right corner)

	if (containedInPoly2D(NUMVERT, polyUV, CPuv))
		return 0.f; //CASE 0: The point lies inside the projected 2D quad.
	if (bContainedOnly) //Early out, if only interested in checking for CONTAINED
		return INFINITY; //NOT contained
	//The remaining code is only to determine the point's outer distance away from the cone's surface.
	float fDist = INFINITY; //The final return value will be stored here
	//Forget about trying to determine which point the line is closest to, and just compute all 3 line-distances, EASY!
	fDist = distanceToSegment2D(CPuv, polyUV[0], polyUV[1]); //Face edge
	fDist = min(fDist, distanceToSegment2D(CPuv, polyUV[2], polyUV[1])); //Top circle's diameter line
	fDist = min(fDist, distanceToSegment2D(CPuv, polyUV[3], polyUV[0])); //Bottom circle's diameter line
		//fDist = min(fDist, distanceToSegment2D(CPuv, (float2)(0.f, 0.f), (float2)(0.f, lenV))); //DEBUG)only; the v)axis connecting both circles

	/*float3 poly[NUMVERT]; //Create the ABCD points of the 2D quad in UV coordinates. They will ALWAYS be oriented the same way in 2D, by construction!
	poly[0] = R[0];		//a (bottom-left corner)
	poly[1] = CTop + R[1];	//b (top-left corner)
	//By construction, the point will always be on the A-B size of the v-axis, not on the C-D side, But we still need all 4 points for a "contains" operation!
	poly[2] = CTop - R[1];	//c (top-right corner)
	poly[3] = -R[0];			//d (bottom right corner)
	fDist = distanceToSegment3D(point, poly[0], poly[1]); //Face edge
	fDist = min(fDist, distanceToSegment3D(point, poly[2], poly[1])); //Top circle's diameter line
	fDist = min(fDist, distanceToSegment3D(point, poly[3], poly[0])); //Bottom circle's diameter line*/
	//fDist = distance(point, (float3)(0,0,0)); //Bottom circle's diameter line
	return fDist * srcMtxInv[32];
}
