#version 130

const float PI = 3.14159265359;
const float TWOPI = 2.0 * PI;
const float PI_OVER_2 = PI / 2.0;
const float EPSILON = 0.000001;
const float SQRT_2 = 1.4142;

const int LOW_RES = 0;
const int HIGH_RES = 1;

//List of textures, which are defined by a single array in Java TEXT_INDEX_COLOR, TEXT_INDEX_DEPTH,:
uniform sampler2D u_texLowResSide; //The color image, RGB only?
uniform sampler2D u_texMipMapVignette; //The current mip-map block of the full-sized color image corresponding to this pixel's "mesh".
// uniform sampler2D u_texDepth; //The depth buffer with RGBA info // TODO take it from MipmappedPOV.fs (&refactor)

//uniform float u_fAmplitude; //0-1 for blinking demolition patch
uniform float u_fHighTexMix; //Fade to high res (u_texMipMapVignette), 0 = fully u_texPano, whereas 1 = fully u_texMipMapVignette.

//uniform float u_fDoDepth; //0/1 boolean float indicating whether "Rendering Mode" is OFF/ON
uniform float u_bDoDebug = 0.0; //0/1 boolean float indicating whether "Rendering Mode" is OFF/ON

//uniform float u_fFOV; //The current Field of View's horizontal angle in radians of the current view.
//uniform ivec2 u_iv2FrameSize; //[horizontal,vertical] span pixel counts.
//uniform float u_fNearClip; //The closest distance from camera that will be rendered
//uniform float u_fFarClip; //The farthest distance from camera that will be rendered

//Image color, u_fContrast, luminosity, u_fBrightness, etc. adjustments:
uniform float u_fIntensityMin; //Rescale the intensity so that this is the lowest possible value.
uniform float u_fIntensityMax; //Rescale the intensity so that this is the highest possible value.
uniform float u_fBrightness; // Sane values are +/-1.2
uniform float u_fContrast; // 0.0 : none, 25.0 : burned
uniform float u_fSaturation; //0.0 : b&w, 25: heatmap.

out vec4 o_v4FragColor; //The final output pixel ARGB that will be displayed on screen

/**
 * @param The desired u_fBrightness offset between +/-1.0
 * @return Translation matrix to change the u_fBrightness of an RGB1 vector4.
 */
mat4 brightnessMatrix( float u_fBrightness ) {
    return mat4( 1, 0, 0, 0,
                 0, 1, 0, 0,
                 0, 0, 1, 0,
                 u_fBrightness, u_fBrightness, u_fBrightness, 1);
}
/**
 * @param The desired u_fContrast between 0-1.
 * @return Generates a transformation matrix that can be applied to an RGB1 vec4 to change its u_fContrast.
 */
mat4 contrastMatrix( float u_fContrast ) {
	float t = ( 1.0 - u_fContrast ) / 2.0;
	//4x4 Scale matrix:
    return mat4( u_fContrast, 0, 0, 0,
                 0, u_fContrast, 0, 0,
                 0, 0, u_fContrast, 0,
                 t, t, t, 1 );
}
/**
 * @param The desired u_fSaturation between 0-1.
 * @return Generates a transformation matrix that can be applied to an RGB1 vec4 to change its u_fSaturation.
 */
mat4 saturationMatrix( float u_fSaturation )
{
    vec3 luminance = vec3( 0.3086, 0.6094, 0.0820 );
    
    float oneMinusSat = 1.0 - u_fSaturation;
    
    vec3 red = vec3( luminance.r * oneMinusSat );
    red+= vec3( u_fSaturation, 0, 0 );
    
    vec3 green = vec3( luminance.g * oneMinusSat );
    green += vec3( 0, u_fSaturation, 0 );
    
    vec3 blue = vec3( luminance.b * oneMinusSat );
    blue += vec3( 0, 0, u_fSaturation );
    //4x4 Rotation&scale matrix:
    return mat4( red,     0,
                 green,   0,
                 blue,    0,
                 0, 0, 0, 1 );
}
/**
 * Entry point for the shader code.
 */
void main()
{
	vec4 v4Color; // prepare output

	//TODO when we want render mode
	//bool bCalculateDepth = u_fDoDepth > 0.5;

	//The current pixel's coordinate in ...
	vec2 tcColor = gl_TexCoord[LOW_RES].xy; //... the color image?
	vec2 tcHigh = gl_TexCoord[HIGH_RES].xy; // ... the mipmap vignette

	v4Color = texture2D(u_texLowResSide, tcColor).rgba; // low res color

	vec2 absolute = 2.0 * abs(0.5 - gl_TexCoord[HIGH_RES].xy); 
	float border  = smoothstep(0.9, 1.0, max(absolute.x, absolute.y));

	vec4 bdColor = vec4(1.0,0.0,0.0,0.0);
	vec4 v4ColorHigh;
	vec4 v4ColorFinal; //Show the regular un-patched color
	if (u_fHighTexMix > 0){
		v4ColorHigh = texture2D(u_texMipMapVignette, tcHigh).rgba;
		bdColor = vec4(0.0,1.0,0.0,0.0);
		v4ColorFinal = mix(v4Color, v4ColorHigh, u_fHighTexMix); //blend-in the high-res
	}else{
		v4ColorHigh = vec4(0.0);// no texture in u_texMipMapVignette.
		bdColor = vec4(1.0,0.0,0.0,0.0);
		v4ColorFinal = v4Color; //Show the regular un-patched color
	}
	if(u_bDoDebug > 0.0)
		v4ColorFinal = mix(v4ColorFinal, bdColor, 0.5 * border); //Show the regular un-patched color

	o_v4FragColor = 1.0 //Transform the color by the brightness, contrast, saturation, and intensity clamp, all specified by user.
		  * brightnessMatrix(u_fBrightness)
          * contrastMatrix(u_fContrast)
          * saturationMatrix(u_fSaturation)
          * vec4( clamp( (v4ColorFinal.rgb - u_fIntensityMin) / (u_fIntensityMax - u_fIntensityMin), 0.0, 1.0 ) , 1.0);
	
	gl_FragDepth = 	gl_FragCoord.z;
}
