/*Some notes about OpenCL.
 * Many GPU's (Quadro included) use little-Endian byte ordering, and Java uses Big-Endian by default, luckily the
 * Jogamp library already converts the CLBuffers whenever necessary; however, sending custom raw byte data requires
 * paying attention to this.
 */

#ifndef __OPENCL_VERSION__ //This should fail during an actual openCL compile, used only to trick Eclipse into syntax highlighting this file as "C" code.
#define __kernel
#define __global
#define kernel
#define global
#define constant
#define local
#define float2 float
#define int3 int
#define float3 float
#define float4 float
#define uchar4 char
#endif

/**TODO: Some very simple shapes don't use a matrix transformation, ensure that they are working properly in this global coordinate system.
 * DONE: Spheres use matrix now
 * 
 * Check for point containment (if direction==NULL) or line intersection (if direction!=NULL) with a list of volumes.
 * Returns the distance to the ray intersection of the rear-face of the closest volume. at [0,180] degrees always positive.
 * If the point lies within any of the volumes, then we must get the distance of the rear-face closest
 * @param volumeParamBuffer A float stream containing floats/ints (all 4 byte aligned, very important!) information about all volumes serialized (concatenated) into a single "stream" buffer.
 * @param volumeFlagBuffer An int buffer containing a 0-1 binary flag indicating whose volume has to be processed by the kernel.
 * @param cellAddressBuff [Only used when at least one of the volumes is a mesh] An int buffer containing the cell's poly-group, points to the group address in the buffPolyGroups.
 * @param buffPolyGroups [Only used when at least one of the volumes is a mesh] An int buffer containing the indices of all the polygons [N, poly0, poly1, ..., polyN] in volumeParamBuffer.
 * @param coords The 3D point (could be global or local use srcMtxInv if it's local) to check if it resides inside the volumes list., or the eye of the ray tracing.
 * @param iProject The project index related to the input octree points, determines which octree-specific matrices to use so the 3D points can be transformed to item-local coordinate system.
 * @param listMtxs A list of [two 4x4 matrices (inv then forward) and 1 scale factor], 1 set for each volume. Inverse transformation converts global to local "coords" *if necessary*
 * (only needed in some cases like a Mesh or primitives).
 * @param fTolerance The max distance to consider a collision occurring.
 * @param fThreshold Maximum distance for colormap.
 * @return Distance to nearest volume collision in meters, or 0 if contained. Negative values indicate an error code.
 */
float collidesWithVolume(constant float *volumeParamBuffer, constant int *volumeFlagBuffer,
		constant float *cellAddressBuff, constant int *buffPolyGroups,
		const float3 coords, const int iProject, constant float *listMtxs, const float fTolerance, const float fThreshold) {
	uint pVPB = 0; //START of volumeParamBuffer to step through the stream for ALL volumes.
	uint pCIB = 0; //START of cellAddressBuff to step through the stream for SOME volumes that needs it (not all volumes will)
	uint pPIB = 0; //START of buffPolyGroups to step through the stream for SOME volumes that needs it (not all volumes will)
	int numVols = readInt(&pVPB, volumeParamBuffer); //Read first int, containing number of shapes (1-127).
	const bool bContainedOnly = fThreshold <= 0.f; //If the threshold is larger than 0, all volumes will attempt to early-out after determining whether contained only. otherwise compute distanceTo too.
	if (numVols < 1 /*|| numVols > 30*/)
		return -1.f; //Error: May have read the buffer incorrectly
	if (iProject < 0 || iProject > 10)
		return -2.f; //Error: May have been given an invalid project ID
	uint pMXB = iProject*(numVols*MTX_CHUNK_SIZE); //START of listMtxs to step through the stream for each volume (but ONLY for a SINGLE ACTIVE project).
	float fOut, fFinal;
	fOut = fFinal = INFINITY; //Collision distance is infinite by default (no collision with ANY volumes)

	for (uint v = 0; v < numVols; v++) { //Cycle through all volumes (1-based index), reading the volumeParamBuffer stream
	
		int volType = readInt(&pVPB, volumeParamBuffer); //read 32-bit integer, containing type (class_id) of current volume
		int volOper = readInt(&pVPB, volumeParamBuffer); //read 32-bit integer, containing operation flag type of current volume
		int chunkSizeVPB = readInt(&pVPB, volumeParamBuffer); //read 32-bit integer, containing length of the current volume (in 32-bit increments)
		constant float *volParamBuffStart = (volumeParamBuffer + pVPB); //Points to the beginning of the volume info, each touches() method will assume it's starting from this address
	
		uint pVolInd = v;
		int volFlag = readInt(&pVolInd, volumeFlagBuffer);//Read the flag for this volume (1 : needs to be processed, 0 : ignore this volume
		if (volFlag == 0)
		{
			switch (volType) {
			case 3416: 
				{ 
					//RvMeshItem
					int3 vParts = readInt3(&pCIB, cellAddressBuff); //Spatial index
					int nParts = vParts.x * vParts.y * vParts.z; //Number of cell partitions
					int chunkSizePIB = readIntI(&pPIB, buffPolyGroups); //read 32-bit integer, containing length of the current volume's chunk in cellAddressBuff (excludes the length var itself)
					pCIB += nParts; //Move onto the next volume
					pPIB += chunkSizePIB; //Move onto the next volume
				}
				break;
			default:
				break;
			}
		}
		else
		{
			switch (volType) {
			case 3105: { //RvLineItem (Poly Line Item)
				fOut = distanceToPolyLine3D(volParamBuffStart, coords);
			}
			break;
			case 3201: { //RvPointItem
				uint pVPBtemp = pVPB;
				float3 point = readFloat3(&pVPBtemp, volumeParamBuffer); //Spatial index
				fOut = length(point - coords);
			}
			break;
			case 3404: //RvPointCloudItem
			case 3405: //RvPointCloudSphereItem
			case 3406: //RvPointCloudTopViewItem
			case 3407: //RvPointCloudLineTopViewItem
				//All point cloud types are the same, just loop through a list of points.
				fOut = distanceToCloud(volParamBuffStart, coords);
			break;
			case 3408: //RvSphereItem, Returns distance to surface or 0 if contained.
				fOut = distanceToSphere(volParamBuffStart, coords, listMtxs + pMXB);
			break;
			case 3410: //RvCircularTorusItem
				fOut = distanceToCircularTorus(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3412: //RvRectangularTorusItem
				fOut = distanceToRectangularTorus(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3413: //RvGenericConeItem
				fOut = distanceToCone(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3414: //RvDishItem
				fOut = distanceToDish(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3415: //RvPortionSphereItem
				fOut = distanceToPortionSphere(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3416: { //RvMeshItem, checks contained and distance to all polygons
				int3 vParts = readInt3(&pCIB, cellAddressBuff); //Spatial index
				int nParts = vParts.x * vParts.y * vParts.z; //Number of cell partitions
				float3 vWidths = readFloat3(&pCIB, cellAddressBuff); //Spatial width of each cell
				int chunkSizePIB = readIntI(&pPIB, buffPolyGroups); //read 32-bit integer, containing length of the current volume's chunk in cellAddressBuff (excludes the length var itself)
						constant int* cellIndBuffStart = (constant int*)cellAddressBuff + pCIB;//Get start address of this volume's cell pointers to groups
				constant int *polyGroupBuffStart = buffPolyGroups + pPIB; //Get start address of this volumes's polygon groups
				fOut = distanceToMesh(volParamBuffStart, cellIndBuffStart, polyGroupBuffStart, vParts, vWidths, coords, listMtxs+pMXB, fTolerance, fThreshold);
				//fOut = 0.1f;
				pCIB += nParts; //Move onto the next volume
				pPIB += chunkSizePIB; //Move onto the next volume
			}
			break;
			case 3418: //RvPyramidItem
				fOut = distanceToPyramid(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3419: //RvGenericExtrudedPolygonItem
				fOut = distanceToExtPoly(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3420: { //RvGenericSegmentItem
				uint pVPBtemp = pVPB;
				float3 pointA = readFloat3(&pVPBtemp, volumeParamBuffer); //First endpoint
				float3 pointB = readFloat3(&pVPBtemp, volumeParamBuffer); //2nd endpoint
				fOut = distanceToSegment3D(coords, pointA, pointB);
			}
			break;
			case 3421: break;//RvGenericPolygonItem
				//case 3422: break;//RvGenericRectangleItem
				case 3423: break;//RvTubeItem
	//			case 3402: //RvCylinderItem (Deprecated)
	//			case 3403: //RvCylinderAutoFitItem (Deprecated)
			case 3430: //RvGenericCylinderItem
				fOut = distanceToCylinder(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			case 3431: //RvGenericBoxItem
				fOut = distanceToBox(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
			break;
			//default: //Error: Invalid volume type, skip it.
				//continue;
			}
			
			if (fOut < fFinal) //Keep the smallest distance from each volume
				fFinal = fOut;
		}
		pVPB += chunkSizeVPB; //Jump to the next volume in the volumeParamBuffer.
		pMXB += MTX_CHUNK_SIZE; //Jump to the next volume in the matrix buffer.
		
		if (fabs(fFinal) < fTolerance) //No need to continue - The point is inside current volume
			break;
	} //Volume list loop
	return fFinal; //Distance to nearest volume collision
}


float collidesWithVolume_LEGACY(constant float *volumeParamBuffer,
		constant float *cellAddressBuff, constant int *buffPolyGroups,
		const float3 coords, const int iProject, constant float *listMtxs, const float fTolerance, const float fThreshold) {
	uint pVPB = 0; //START of volumeParamBuffer to step through the stream for ALL volumes.
	uint pCIB = 0; //START of cellAddressBuff to step through the stream for SOME volumes that needs it (not all volumes will)
	uint pPIB = 0; //START of buffPolyGroups to step through the stream for SOME volumes that needs it (not all volumes will)
	int numVols = readInt(&pVPB, volumeParamBuffer); //Read first int, containing number of shapes (1-127).
	const bool bContainedOnly = fThreshold <= 0.f; //If the threshold is larger than 0, all volumes will attempt to early-out after determining whether contained only. otherwise compute distanceTo too.
	if (numVols < 1 /*|| numVols > 30*/)
		return -1.f; //Error: May have read the buffer incorrectly
	if (iProject < 0 || iProject > 10)
		return -2.f; //Error: May have been given an invalid project ID
	uint pMXB = iProject*(numVols*MTX_CHUNK_SIZE); //START of listMtxs to step through the stream for each volume (but ONLY for a SINGLE ACTIVE project).
	float fOut, fFinal;
	fOut = fFinal = INFINITY; //Collision distance is infinite by default (no collision with ANY volumes)

	for (uint v = 0; v < numVols; v++) { //Cycle through all volumes (1-based index), reading the volumeParamBuffer stream
	
		int volType = readInt(&pVPB, volumeParamBuffer); //read 32-bit integer, containing type (class_id) of current volume
		int volOper = readInt(&pVPB, volumeParamBuffer); //read 32-bit integer, containing operation flag type of current volume
		int chunkSizeVPB = readInt(&pVPB, volumeParamBuffer); //read 32-bit integer, containing length of the current volume (in 32-bit increments)
		constant float *volParamBuffStart = (volumeParamBuffer + pVPB); //Points to the beginning of the volume info, each touches() method will assume it's starting from this address
	
		switch (volType) {
		case 3105: { //RvLineItem (Poly Line Item)
			fOut = distanceToPolyLine3D(volParamBuffStart, coords);
		}
		break;
		case 3201: { //RvPointItem
			uint pVPBtemp = pVPB;
			float3 point = readFloat3(&pVPBtemp, volumeParamBuffer); //Spatial index
			fOut = length(point - coords);
		}
		break;
		case 3404: //RvPointCloudItem
		case 3405: //RvPointCloudSphereItem
		case 3406: //RvPointCloudTopViewItem
		case 3407: //RvPointCloudLineTopViewItem
			//All point cloud types are the same, just loop through a list of points.
			fOut = distanceToCloud(volParamBuffStart, coords);
		break;
		case 3408: //RvSphereItem, Returns distance to surface or 0 if contained.
			fOut = distanceToSphere(volParamBuffStart, coords, listMtxs + pMXB);
		break;
		case 3410: //RvCircularTorusItem
			fOut = distanceToCircularTorus(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3412: //RvRectangularTorusItem
			fOut = distanceToRectangularTorus(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3413: //RvGenericConeItem
			fOut = distanceToCone(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3414: //RvDishItem
			fOut = distanceToDish(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3415: //RvPortionSphereItem
			fOut = distanceToPortionSphere(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3416: { //RvMeshItem, checks contained and distance to all polygons
			int3 vParts = readInt3(&pCIB, cellAddressBuff); //Spatial index
			int nParts = vParts.x * vParts.y * vParts.z; //Number of cell partitions
			float3 vWidths = readFloat3(&pCIB, cellAddressBuff); //Spatial width of each cell
			int chunkSizePIB = readIntI(&pPIB, buffPolyGroups); //read 32-bit integer, containing length of the current volume's chunk in cellAddressBuff (excludes the length var itself)
					constant int* cellIndBuffStart = (constant int*)cellAddressBuff + pCIB;//Get start address of this volume's cell pointers to groups
			constant int *polyGroupBuffStart = buffPolyGroups + pPIB; //Get start address of this volumes's polygon groups
			fOut = distanceToMesh(volParamBuffStart, cellIndBuffStart, polyGroupBuffStart, vParts, vWidths, coords, listMtxs+pMXB, fTolerance, fThreshold);
			//fOut = 0.1f;
			pCIB += nParts; //Move onto the next volume
			pPIB += chunkSizePIB; //Move onto the next volume
		}
		break;
		case 3418: //RvPyramidItem
			fOut = distanceToPyramid(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3419: //RvGenericExtrudedPolygonItem
			fOut = distanceToExtPoly(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3420: { //RvGenericSegmentItem
			uint pVPBtemp = pVPB;
			float3 pointA = readFloat3(&pVPBtemp, volumeParamBuffer); //First endpoint
			float3 pointB = readFloat3(&pVPBtemp, volumeParamBuffer); //2nd endpoint
			fOut = distanceToSegment3D(coords, pointA, pointB);
		}
		break;
		case 3421: break;//RvGenericPolygonItem
			//case 3422: break;//RvGenericRectangleItem
			case 3423: break;//RvTubeItem
//			case 3402: //RvCylinderItem (Deprecated)
//			case 3403: //RvCylinderAutoFitItem (Deprecated)
		case 3430: //RvGenericCylinderItem
			fOut = distanceToCylinder(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		case 3431: //RvGenericBoxItem
			fOut = distanceToBox(volParamBuffStart, coords, listMtxs+pMXB, bContainedOnly);
		break;
		//default: //Error: Invalid volume type, skip it.
			//continue;
		}
		
		if (fOut < fFinal) //Keep the smallest distance from each volume
			fFinal = fOut;
		
		pVPB += chunkSizeVPB; //Jump to the next volume in the volumeParamBuffer.
		pMXB += MTX_CHUNK_SIZE; //Jump to the next volume in the matrix buffer.
		
		if (fabs(fFinal) < fTolerance) //No need to continue - The point is inside current volume
			break;
	} //Volume list loop
	return fFinal; //Distance to nearest volume collision
}
/*----------------------------------------------------------*//**
 * Provide a list of 3D points and returns a list of the colors depending on their distance to nearest collision volume.
 * @param volumeParamBuffer A byte stream containing information about all volumes serialized (concatenated) into a single "stream" buffer.
 * @param volumeFlagBuffer An int buffer containing a 0-1 binary flag indicating whose volume has to be processed by the kernel.
 * @param cellAddressBuff [Only used when at least one of the volumes is a mesh] An int buffer containing the cell's poly-group, points to the group address in the buffPolyGroups.
 * @param buffPolyGroups [Only used when at least one of the volumes is a mesh] An int buffer containing the indices of all the polygons [N, poly0, poly1, ..., polyN] in volumeParamBuffer.
 * @param listPoints List of 3D points [x,y,z,color], 1 per worker.
 * @param listCollides Output list of float distances for each point (0=NO Collision, 1=touching) for each point touching all volumes. Use the threshold distance to scale it into meters.
 * @param iProject The project index related to the input octree points, determines which octree-specific matrices to use so the 3D points can be transformed to item-local coordinate system.
 * @param listMtxs A list of [two 4x4 matrices (inv then forward) and 1 scale factor], 1 set for each volume. Inverse transformation converts global to local "coords" *if necessary*
 * (only needed in some cases like a Mesh or primitives).
 * @param nChunk Number of items that a single worker should handle.
 * @param nPoints Length of listPoints and listColors.
 * @param fTolerance The max distance to consider a collision occurring.
 * @param fThreshold Maximum distance for colormap.
 * @return (output written to listCollides) Distance to nearest volume collision. Negative values indicate an error code (and should be a byte value) ranging from -128 to -1.
 *//*-----------------------------------------------------------*/
kernel void identifyCollisions(constant float *volumeParamBuffer, constant int *volumeFlagBuffer,
		constant float *cellAddressBuff, constant int *buffPolyGroups,
		global const float4 *listPoints, global float *listCollides, const int iProject, constant float *listMtxs, const int nChunk, const int nPoints, const float fTolerance, const float fThreshold) {
	const int iWorker = get_global_id(0); //The worker ID, note that it will only span HALF the image domain, each worker is responsible for 2 pixels each.

	for (int i = iWorker * nChunk; i < (iWorker + 1) * nChunk; i++) {
		if (i >= nPoints) //skip if out of bounds, node may be smaller than the rounded-up worker count (because of fixed group sizes).
			return;
		const float3 coords = listPoints[i].xyz; //Copy the point for the current worker.
		float minDist = collidesWithVolume(volumeParamBuffer, volumeFlagBuffer, cellAddressBuff, buffPolyGroups, coords, iProject, listMtxs, fTolerance, fThreshold);//determine if collides with a list of 3D volumes, returns the closest distance.
		//float color = listPoints[i].w;
		float fOut; //0-1 only! Inverted distance where 1.0 indicates touching collision, 0=No collision.
		if (minDist >= 0.f) { //No error
			if (minDist <= fThreshold) //A flag must appear within the colormap range.
				fOut = (fThreshold - minDist) / fThreshold; //The closer the distance, the redder it is 0.0->1.0 (using 0-1 range for distance, reserving the rest for other flags)
			else //NO COLLISION occurred
				fOut = 0.f;
		}
		else //convert to error code:
			fOut = fThreshold;
		listCollides[i] = fOut;
	}
}
kernel void identifyCollisions_LEGACY(constant float *volumeParamBuffer,
		constant float *cellAddressBuff, constant int *buffPolyGroups,
		global const float4 *listPoints, global float *listCollides, const int iProject, constant float *listMtxs, const int nChunk, const int nPoints, const float fTolerance, const float fThreshold) {
	const int iWorker = get_global_id(0); //The worker ID, note that it will only span HALF the image domain, each worker is responsible for 2 pixels each.

	for (int i = iWorker * nChunk; i < (iWorker + 1) * nChunk; i++) {
		if (i >= nPoints) //skip if out of bounds, node may be smaller than the rounded-up worker count (because of fixed group sizes).
			return;
		const float3 coords = listPoints[i].xyz; //Copy the point for the current worker.
		float minDist = collidesWithVolume_LEGACY(volumeParamBuffer, cellAddressBuff, buffPolyGroups, coords, iProject, listMtxs, fTolerance, fThreshold);//determine if collides with a list of 3D volumes, returns the closest distance.
		//float color = listPoints[i].w;
		float fOut; //0-1 only! Inverted distance where 1.0 indicates touching collision, 0=No collision.
		if (minDist >= 0.f) { //No error
			if (minDist <= fThreshold) //A flag must appear within the colormap range.
				fOut = (fThreshold - minDist) / fThreshold; //The closer the distance, the redder it is 0.0->1.0 (using 0-1 range for distance, reserving the rest for other flags)
			else //NO COLLISION occurred
				fOut = 0.f;
		}
		else //convert to error code:
			fOut = fThreshold;
		listCollides[i] = fOut;
	}
}
